/** 
 * Factory to get  Bizerba UposScale Objects Copyright (c) 2007-2011 BIZERBA  
 *  
 * The class delivers UposScale service objects which
 * delegate the service to the native libray BUSCI
 *   
 * @author Reiner Moll 
 * @version 220-1 - 30.01.2014
 * 
 */

package com.bizerba.jpos.scalecsoem.UPOSLib;

import jpos.JposConst;
import jpos.JposException;

import com.bizerba.jpos.scalecsoem.UPOSLib.NativeUposScale;


public class UposScale {
	/** In case of approval relevant changes increase this ID */
	public static final int APPROVAL_ID = 100;
	/** Object to native Interface and native library BUSCI */	
	protected static IUposScale m_jni = null;
	/** Handle to Object in native library BUSCI */
	private long m_insthdl=-1;
	/** Instant counter to control the exit of the native library BUSCI  */
	private static int m_refs=0;
	/** */
	private boolean m_instanceMustbeReleased = false;
	
	
	/**
	 * @param moduleFileName1 Path native library
	 * @param moduleFileName2 Path for JAR archive
	 * @param approvalJava ID for approval in mode Scale OEM
	 * @param approvalJavaCS ID for approval in mode CS scale directly 
	 * @param checksum checksum for jar archive 
	 * @param correction Polynom 16 correction value
	 * @param configfile full name for the initialization file
	 * @param updateDTCCS call back to send DTCCS events
	 *
	 * @return The function delivers an instance of an UposScale object
	 * 
	 * @remarks The native BUSCI library will be initialized with the first instance
	 * 
	 */
	public static synchronized UposScale getInstance(String moduleFileName1, String modulFileName2, int approvalJava, String configfile, IUpdateDTCCS updateDTCCS, boolean useZMQ, boolean autoStartZMQ,String port, String publisherPort, String version) throws JposException
	{
	
		if(m_jni == null)
		{
			if(true == useZMQ)
			{
				m_jni = new UposScaleZMQClient(moduleFileName1, port, publisherPort, autoStartZMQ, updateDTCCS);
			}
			else
			{
				try
				{
					m_jni = new NativeUposScale(moduleFileName1, updateDTCCS);
				}
				catch (UnsatisfiedLinkError e)
				{
					System.out.println("can't load native library "+ e.getMessage());
					throw new JposException(JposConst.JPOS_E_EXTENDED,(int)10000,
							"can't load native library ");
				}
			}
		}
		if ((configfile!=null)&&(!configfile.isEmpty()))
		{
			String[] errorText = new String[1];
			long ret = m_jni.OptionalPreInit(configfile,errorText);			
			testResult(ret,errorText[0]);
		}
		return new UposScale(moduleFileName1, modulFileName2, approvalJava,version);
	};
	
	/**
	 * Releases the object 
	 * 
	 * @remarks In case the last object is released,
	 * the BUSCI library will be exited
	 *  
	 */	
	public synchronized void releaseInstance()
	{
		if(m_instanceMustbeReleased==true)
		{
			m_instanceMustbeReleased = false;
			if(m_refs != 0)
			{
				m_refs--;
				if (m_refs==0)
				{
					m_jni.Exit();
					m_jni = null;					
				}
			}
		}
	}
	
	/**
	 * @param moduleFileName1 Path for Native library
	 * @param moduleFileName2 Path for jar archive
	 * @param checksum checksum for jar archive 
	 * @param correction Polynom 16 correction value
	 * 
	 * @remarks constructs an UposScale object 
	 * The native BUSCI library will be initialized with the first instance
	 * 
	 */	
	protected UposScale(String moduleFileName1, String moduleFileName2, int approvalJava , String version) throws JposException{	
		if (m_refs==0)
		{
			String[] errorText = new String[1];
			long ret=m_jni.Init(moduleFileName1, moduleFileName2 , approvalJava, version, errorText);
			testResult(ret,errorText[0]);
		}
		m_instanceMustbeReleased = true;
		m_refs++;
	}
	
	/**
	 * destroys the UposScale object
	 * 
	 * @remarks In case the last object is destroyed,
	 * the BUSCI library will be exited
	 * only called when there is no ref any more 
	 */	
	protected void finalize() throws Throwable 
	{
		try
		{
			releaseInstance();
		}
		finally
		{
			super.finalize();
		}
	}
	
	/**
	 * Opens a scale device
	 * name is the physical device name
	 * 
	 * @param name
	 * @throws JposException
	 */	
	public void open(String name)throws JposException {
		long l_insthdl[]= new long[1];
		l_insthdl[0]=m_insthdl;
				
		String[] errorText = new String[1];
		long ret=m_jni.Open(name, l_insthdl,errorText);
		testResult(ret, errorText[0]);
		m_insthdl=l_insthdl[0];//only overwrite on success			
	}
	
	/**
	 * Close scale device
	 * @throws JposException
	 * 
	 */
	public void close() throws JposException{
		String[] errorText = new String[1];
		long ret=m_jni.Close(m_insthdl,errorText);
		testResult(ret, errorText[0]);
		m_insthdl=-1;//only overwrite on success		
	}
	
	/**
	 * Claim device
	 * @param timeout_in_ms
	 * @throws JposException
	 */		
	public void claim (long timeout_in_ms) throws JposException
	{
		String[] errorText = new String[1];
		long ret=m_jni.Claim(m_insthdl, timeout_in_ms, errorText);
		testResult(ret, errorText[0]);
	}
	
	/**
	 * Release device 
	 * @throws JposException
	 */
	public void releaseDevice () throws JposException
	{
		String[] errorText = new String[1];		
		long ret=m_jni.ReleaseDevice(m_insthdl, errorText);
		testResult(ret, errorText[0]);
	}
	
	/**
	 *  
	 * @return true if device is claimed
	 * @throws JposException
	 */
	public boolean getClaimed () throws JposException
	{
		boolean[] claimed = new boolean[1];
		String[] errorText = new String[1];
		long ret=m_jni.GetClaimed(m_insthdl, claimed, errorText);
		testResult(ret, errorText[0]);
		return claimed[0];
	}
	
	/**
	 * 
	 * @return true if device is enabled
	 * @throws JposException
	 */
	public boolean getDeviceEnabled () throws JposException
	{
		boolean[] enabled = new boolean[1];
		String[] errorText = new String[1];
		long ret=m_jni.GetDeviceEnabled(m_insthdl, enabled, errorText);
		testResult(ret, errorText[0]);
		return enabled[0];		
	}
	
	/** 
	 * Enable disable  the devices
	 * @param enabled
	 * @throws JposException
	 */
	public void setDeviceEnabled (boolean enabled) throws JposException
	{
		String[] errorText = new String[1];
		long ret=m_jni.SetDeviceEnabled(m_insthdl, enabled, errorText);
		testResult(ret, errorText[0]);		
	}
	
	/**
	 * 
	 * @return scale state
	 * @throws JposException
	 */
	public int getState () throws JposException
	{
		long[] state = new long[1];
		String[] errorText = new String[1];
		long ret=m_jni.GetState(m_insthdl, state, errorText);
		testResult(ret, errorText[0]);
		return (int)state[0];		
	}
	
	/**
	 * Direct IO commands 
	 * @param id
	 * @param data
	 * @param datastring
	 * @throws JposException
	 */
	public void directIO(long id, long data[], Object datastring[]) throws JposException
	{
		String[] errorText = new String[1];
		long ret=m_jni.DirectIO(m_insthdl,id, data, datastring, errorText);
		testResult(ret, errorText[0]);			
	}
	
	/**
	 * Set sync/async mode
	 * @param asyncMode
	 * @throws JposException
	 */
	public void setAsyncMode(boolean  asyncMode) throws JposException
	{
		String[] errorText = new String[1];
		long ret=m_jni.SetAsyncMode(m_insthdl, asyncMode, errorText);
		testResult(ret, errorText[0]);		
	}
	
	/**
	 * 
	 * @return true if async mode is active 
	 * @throws JposException
	 */
	public boolean getAsyncMode() throws JposException
	{
		boolean[] asyncmode = new boolean[1];
		String[] errorText = new String[1];
		long ret=m_jni.GetAsyncMode(m_insthdl, asyncmode, errorText);
		testResult(ret, errorText[0]);
		return asyncmode[0];		
	}
	
	/**
	 * 
	 * @return true if display 
	 * @throws JposException
	 */
	public boolean capDisplay() throws JposException
	{
		boolean[] capdisplay = new boolean[1];
		String[] errorText = new String[1];
		long ret=m_jni.CapDisplay(m_insthdl, capdisplay, errorText);
		testResult(ret, errorText[0]);
		return capdisplay[0];	
	}
	
	/**
	 * 
	 * @return true if display text possible
	 * @throws JposException
	 */
	public boolean capDisplayText() throws JposException
	{
		boolean[] capdisplaytext = new boolean[1];
		String[] errorText = new String[1];
		long ret=m_jni.CapDisplayText(m_insthdl, capdisplaytext, errorText);
		testResult(ret, errorText[0]);
		return capdisplaytext[0];
	}
	
	/**
	 * 
	 * @return true if price calculating active
	 * @throws JposException
	 */
	public boolean capPriceCalculating() throws JposException
	{
		boolean[] cappricecalculating = new boolean[1];
		String[] errorText = new String[1];
		long ret=m_jni.CapPriceCalculating(m_insthdl, cappricecalculating, errorText);
		testResult(ret, errorText[0]);
		return cappricecalculating[0];
	}
	
	/**
	 * 
	 * @return true if tare weight capability
	 * @throws JposException
	 */
	public boolean capTareWeight() throws JposException
	{
		boolean[] captareweight = new boolean[1];
		String[] errorText = new String[1];
		long ret=m_jni.CapTareWeight(m_insthdl, captareweight, errorText);
		testResult(ret, errorText[0]);
		return captareweight[0];		
	}
	
	/**
	 * 
	 * @return true if zero scale capability
	 * @throws JposException
	 */
	public boolean capZeroScale() throws JposException
	{
		boolean[] capzeroweight = new boolean[1];
		String[] errorText = new String[1];
		long ret=m_jni.CapZeroScale(m_insthdl, capzeroweight, errorText);
		testResult(ret, errorText[0]);
		return capzeroweight[0];	
	}
	
	/**
	 * 
	 * @return the maximum number of characters
	 * @throws JposException
	 */
	public int getMaxDisplayTextChars() throws JposException
	{
		long[] maxdisplaytextchars = new long[1];
		String[] errorText = new String[1];
		long ret=m_jni.GetMaxDisplayTextChars(m_insthdl, maxdisplaytextchars, errorText);
		testResult(ret, errorText[0]);
		return (int)maxdisplaytextchars[0];	
	}
	
	/**
	 * 
	 * @return the sales price
	 * @throws JposException
	 */
	public long getSalesPrice() throws JposException
	{
		long[] salesprice = new long[1];
		String[] errorText = new String[1];
		long ret=m_jni.GetSalesPrice(m_insthdl, salesprice, errorText);
		testResult(ret, errorText[0]);
		return salesprice[0];	
	}
	
	/**
	 * Set the tare weight 
	 * @param tareWeight
	 * @throws JposException
	 */
	public void setTareWeight(int  tareWeight) throws JposException
	{
		String[] errorText = new String[1];
		long ret=m_jni.SetTareWeight(m_insthdl, tareWeight, errorText);
		testResult(ret, errorText[0]);		
	}
	
	/***
	 * 
	 * @return the active tare weight
	 * @throws JposException
	 */
	public int getTareWeight() throws JposException
	{
		long[] tareweight = new long[1];
		String[] errorText = new String[1];
		long ret=m_jni.GetTareWeight(m_insthdl, tareweight, errorText);
		testResult(ret, errorText[0]);
		return (int)tareweight[0];	
	}
	
	/**
	 *  Set unit price 
	 * @param unitPrice
	 * @throws JposException
	 */
	public void setUnitPrice(long unitPrice) throws JposException
	{
		String[] errorText = new String[1];
		long ret=m_jni.SetUnitPrice(m_insthdl, unitPrice, errorText);
		testResult(ret, errorText[0]);	
	}
	
	/**
	 * 
	 * @return the actual unit price
	 * @throws JposException
	 */
	public long getUnitPrice() throws JposException
	{
		long[] unitprice = new long[1];
		String[] errorText = new String[1];
		long ret=m_jni.GetUnitPrice(m_insthdl, unitprice, errorText);
		testResult(ret, errorText[0]);
		return unitprice[0];		
	}
	
	/**
	 * 
	 * @return the weight unit
	 * @throws JposException
	 */
	public int getWeightUnit() throws JposException
	{
		long[] weightunit = new long[1];
		String[] errorText = new String[1];
		long ret=m_jni.GetWeightUnit(m_insthdl, weightunit, errorText);
		testResult(ret, errorText[0]);
		return (int)weightunit[0];	
	}
	
	/**
	 * Set the display text
	 * @param text
	 * @throws JposException
	 */
	public void setDisplayText(String text) throws JposException
	{
		String[] errorText = new String[1];
		long ret=m_jni.SetDisplayText(m_insthdl, text, errorText);
		testResult(ret, errorText[0]);
	}
	
	/**
	 * 
	 * @param timeout
	 * @return the read weight
	 * @throws JposException
	 */
	public int readWeight(long timeout) throws JposException
	{
		long[] readweight = new long[1];
		String[] errorText = new String[1];
		long ret=m_jni.ReadWeight(m_insthdl, readweight, timeout ,errorText);
		testResult(ret, errorText[0]);
		return (int)readweight[0];	
	}
	
	/**
	 * Zero the scale
	 * 
	 * @throws JposException
	 */
	public void zeroScale() throws JposException
	{
		String[] errorText = new String[1];
		long ret=m_jni.ZeroScale(m_insthdl,errorText);
		testResult(ret, errorText[0]);		
	}
	
	/**
	 * 
	 * @return the stable live weight
	 * @throws JposException
	 */
	public int scaleLiveWeight() throws JposException
	{
		long[] liveweight = new long[1];
		String[] errorText = new String[1];
		long ret=m_jni.ScaleLiveWeight(m_insthdl, liveweight,errorText);
		testResult(ret, errorText[0]);
		return (int)liveweight[0];			
	}
	
	/**
	 * 
	 * @return the maximum weight
	 * @throws JposException
	 */
	public int getMaximumWeight() throws JposException
	{
		long[] maximumweight = new long[1];
		String[] errorText = new String[1];
		long ret=m_jni.GetMaximumWeight(m_insthdl, maximumweight,errorText);
		testResult(ret, errorText[0]);
		return (int)maximumweight[0];			
	}
	
	/**
	 * Stops an asynchron started command
	 * 
	 * @throws JposException
	 */
	public void killActualCommand()throws JposException
	{
		String[] errorText = new String[1];
		long ret=m_jni.KillActualCommand(m_insthdl,errorText);
		testResult(ret, errorText[0]);	
	}
	
	/**
	 * Tests the return code and creates Exception
	 * 
	 * @param ret  Test ret and creates Exception if ret != JposConst.JPOS_SUCCESS 
	 * @throws JposException
	 * 
	 * @remarks ret high word is errorCode, low word is errorCodeExtended
	 */	
	static protected void testResult(long ret, String errorText )throws JposException
	{
		if(ret!=JposConst.JPOS_SUCCESS)
		{
			// errorCode for UPOS
			int	errorCode; 
		    // extended Error Codes
		    int errorCodeExtended;	
		    errorCode = (int)( ret >> 32);
		    errorCodeExtended = (int)ret;
		    
		    if(errorCodeExtended != 0)
		    {
				throw new JposException(JposConst.JPOS_E_EXTENDED,(int)errorCodeExtended,
						errorText);
		    }
		    else
		    {
				throw new JposException(errorCode,errorText);		    
		    }	    
		}
	}
	
	/**
	 * This method returns the string representing the Device Service
	 * Description
	 * 
	 * @see jpos.services.BaseService#getDeviceServiceDescription()
	 */
	public String getDeviceServiceDescription() throws JposException
	{
		String[] errorText = new String[1];
		String[] devicesDescription = new String[1];
		long ret= m_jni.GetDeviceServiceDescription(m_insthdl, devicesDescription, errorText);
		testResult(ret, errorText[0]);
		return devicesDescription[0];
	}

	/**
	 * This method returns the integer value representing the version number for
	 * Device Service
	 * 
	 * @see jpos.services.BaseService#getDeviceServiceVersion()
	 */
	static public int GetDeviceServiceVersion() throws JposException
	{
		return m_jni.GetDeviceServiceVersion();
	}
	
	/**
	 * This methods makes a check health 
	 * 
	 * @param level
	 * @throws JposException
	 */
	public void checkHealth(long level) throws JposException
	{
		String[] errorText = new String[1];
		long ret=m_jni.CheckHealth(m_insthdl,level,errorText);
		testResult(ret, errorText[0]);
	}
	
	/**
	 * This method delivers the last check health text
	 * 
	 * @return last health text
	 * @throws JposException
	 */	
	public String getHealthText() throws JposException
	{
		String[] errorText = new String[1];
		String[] healthText = new String[1];
		long ret=m_jni.GetHealthText(m_insthdl, healthText, errorText);
		testResult(ret, errorText[0]);
		return healthText[0];		
	}
	
	/* UPOS 1.13 */
	
	/**
	 * 
	 * @return
	 * @throws JposException
	 */
	public boolean getZeroValid() throws JposException {
		// TODO Auto-generated method stub
		String[] errorText = new String[1];
		boolean[] zeroValid = new boolean[1];
		long ret= m_jni.ZeroValid(m_insthdl, zeroValid, errorText);
		testResult(ret, errorText[0]);		
		return zeroValid[0];
	}
	
	/* UPOS 1.14 */
	
	/**
	 * 
	 * @param weightData
	 * @param tare
	 * @param unitPrice
	 * @param unitPriceX
	 * @param weightUnitX
	 * @param weightNumeratorX
	 * @param weightDenominatorX
	 * @param price
	 * @param timeout
	 * @throws JposException
	 */	
	public void doPriceCalculating(int[] weightData, int[] tare, long[] unitPrice,
			long[] unitPriceX, int[] weightUnitX, int[] weightNumeratorX, int[] weightDenominatorX, long[] price,
			int timeout) throws JposException {
		// TODO Auto-generated method stub
		String[] errorText = new String[1];
		long ret= m_jni.DoPriceCalculating(m_insthdl, weightData, tare, unitPrice, unitPriceX, weightUnitX, weightNumeratorX, weightDenominatorX, price, timeout, errorText);
		testResult(ret, errorText[0]);		
	}

	/**
	 * 
	 * @param item
	 * @param freeze
	 * @throws JposException
	 */	
	public void freezeValue(int item, boolean freeze) throws JposException {
		// TODO Auto-generated method stub
		String[] errorText = new String[1];
		long ret=m_jni.freezeValue(m_insthdl, item, freeze, errorText);
		testResult(ret, errorText[0]);	
	}
	
	/**
	 * 
	 * @return
	 * @throws JposException
	 */
	public boolean getCapFreezeValue() throws JposException {
		// TODO Auto-generated method stub
		String[] errorText = new String[1];
		boolean[] capFreezeValue = new boolean[1];
		long ret=m_jni.CapFreezeValue(m_insthdl, capFreezeValue, errorText);
		testResult(ret, errorText[0]);		
		return capFreezeValue[0];
	}

	/**
	 * 
	 * @return
	 * @throws JposException
	 */
	public boolean getCapReadLiveWeightWithTare() throws JposException {
		// TODO Auto-generated method stub
		String[] errorText = new String[1];
		boolean[] capReadLiveWeightWithTare = new boolean[1];
		long ret=m_jni.CapReadLiveWeightWithTare(m_insthdl, capReadLiveWeightWithTare, errorText);
		testResult(ret, errorText[0]);
		return capReadLiveWeightWithTare[0];
	}

	/**
	 * 
	 * @return
	 * @throws JposException
	 */
	public boolean getCapSetPriceCalculationMode() throws JposException {
		// TODO Auto-generated method stub
		String[] errorText = new String[1];
		boolean[] capSetPriceCalculationMode = new boolean[1];
		long ret=m_jni.CapSetPriceCalculationMode(m_insthdl, capSetPriceCalculationMode, errorText);
		testResult(ret, errorText[0]);
		return capSetPriceCalculationMode[0];
	}

	/**
	 * 
	 * @return
	 * @throws JposException
	 */
	public boolean getCapSetUnitPriceWithWeightUnit() throws JposException {
		// TODO Auto-generated method stub
		String[] errorText = new String[1];
		boolean[] capSetUnitPriceWithWeightUnit = new boolean[1];
		long ret=m_jni.CapSetUnitPriceWithWeightUnit(m_insthdl, capSetUnitPriceWithWeightUnit, errorText);
		testResult(ret, errorText[0]);	
		return capSetUnitPriceWithWeightUnit[0];
	}

	/**
	 * 
	 * @return
	 * @throws JposException
	 */
	public boolean getCapSpecialTare() throws JposException {
		// TODO Auto-generated method stub
		String[] errorText = new String[1];
		boolean[] capSpecialTare = new boolean[1];
		long ret=m_jni.CapSpecialTare(m_insthdl, capSpecialTare, errorText);
		testResult(ret, errorText[0]);	
		return capSpecialTare[0];
	}

	/**
	 * 
	 * @return
	 * @throws JposException
	 */
	public boolean getCapTarePriority() throws JposException {
		// TODO Auto-generated method stub
		String[] errorText = new String[1];
		boolean[] capTarePriority = new boolean[1];
		long ret=m_jni.CapTarePriority(m_insthdl, capTarePriority, errorText);
		testResult(ret, errorText[0]);	
		return capTarePriority[0];
	}

	/**
	 * 
	 * @return
	 * @throws JposException
	 */
	public int getMinimumWeight() throws JposException {
		// TODO Auto-generated method stub	
		
		long[] minimumweight = new long[1];
		String[] errorText = new String[1];
		long ret=m_jni.GetMinimumWeight(m_insthdl, minimumweight,errorText);
		testResult(ret, errorText[0]);
		return (int)minimumweight[0];
	}

	/**
	 * 
	 * @param weightData
	 * @param tare
	 * @param timeout
	 * @throws JposException
	 */
	public void readLiveWeightWithTare(int[] weightData, int[] tare, int timeout)
			throws JposException {
		// TODO Auto-generated method stub
		String[] errorText = new String[1];
		long ret=m_jni.ReadLiveWeightWithTare(m_insthdl, weightData, tare, timeout, errorText);
		testResult(ret, errorText[0]);	
	}

	/**
	 * 
	 * @param mode
	 * @throws JposException
	 */
	public void setPriceCalculationMode(int mode) throws JposException {
		// TODO Auto-generated method stub
		String[] errorText = new String[1];
		long ret=m_jni.SetPriceCalculationMode(m_insthdl, mode, errorText);
		testResult(ret, errorText[0]);	
	}

	/**
	 * 
	 * @param mode
	 * @param data
	 * @throws JposException
	 */
	public void setSpecialTare(int mode, int data) throws JposException {
		// TODO Auto-generated method stub
		String[] errorText = new String[1];
		long ret=m_jni.SetSpecialTare(m_insthdl, mode, data, errorText);
		testResult(ret, errorText[0]);	
	}

	/**
	 * 
	 * @param priority
	 * @throws JposException
	 */
	public void setTarePrioity(int priority) throws JposException {
		// TODO Auto-generated method stub
		String[] errorText = new String[1];
		long ret=m_jni.SetTarePrioity(m_insthdl, priority, errorText);
		testResult(ret, errorText[0]);		
	}

	/**
	 * 
	 * @param unitPrice
	 * @param weightUnit
	 * @param weightNumerator
	 * @param weightDenominator
	 * @throws JposException
	 */
	public void setUnitPriceWithWeightUnit(long unitPrice, int weightUnit, int weightNumerator,
			int weightDenominator) throws JposException {
		// TODO Auto-generated method stub
		String[] errorText = new String[1];
		long ret=m_jni.SetUnitPriceWithWeightUnit(m_insthdl, unitPrice, weightUnit, weightNumerator, weightDenominator, errorText);
		testResult(ret, errorText[0]);		
	}
}
	

