/** 
 * Implementation of Bizerba Scale Service 19  
 * Copyright (c) 2007-2011 BIZERBA
 *   
 * @author Reiner Moll 
 * @version 205-1 - 04.10.2011
 * 
 */
package com.bizerba.jpos.scalecsoem;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.Vector;
import java.util.concurrent.Semaphore;
import java.util.logging.ConsoleHandler;
import java.util.logging.Handler;
import java.util.logging.Level;
import java.util.logging.LogManager;
import java.util.logging.Logger;
import java.util.logging.SimpleFormatter;
import java.util.regex.Pattern;

import jpos.JposConst;
import jpos.JposException;
import jpos.ScaleConst;
import jpos.config.JposEntry;
import jpos.config.JposEntryRegistry;
import jpos.events.DataEvent;
import jpos.events.DirectIOEvent;
import jpos.events.ErrorEvent;
import jpos.events.StatusUpdateEvent;
import jpos.loader.JposServiceLoader;
import jpos.services.EventCallbacks;
import jpos.services.ScaleService19;

import com.bizerba.jpos.scalecsoem.UPOSLib.*;


/**
 * The {@code BizerbaCSScaleService19} class implements the
 * {@code ScaleService19} and represents the Device Service part of JavaPOS 1.9
 * scale driver for Bizerba Scale Model CS.
 * 
 * @author Bizerba
 * @version 2.051 5.10.2011 Reiner Moll common native interface for JPOS and OPOS 
 * 
 * @remarks: This module contains approval relevant code                        
 *           If you change approval relevant code you have to increase APPROVAL_ID    
 *           and you have to induce a new authorization
 * 
 * @see ScaleService19
 */
public class BizerbaCSScaleService19 implements ScaleService19, IUpdateDTCCS {
	// JavaPOS common attributes
	/** In case of approval relevant changes increase this ID */
	public static final int APPROVAL_ID = 1;
	public static final String DEFAULT_PHYSICAL_DESCRIPTION = "Bizerba JPOS Scale";	
	protected final String PHYSICAL_DEVICE_NAME = "physicalDeviceName";
	protected final String PHYSICAL_DEVICE_DESCRIPTION = "physicalDeviceDescripton";
	protected final String INI_FILE_KEY = "config_file_with_rw_access_without_extension";
	protected final String LOGGING_PROPERTIES_FILE = "loggingPropertiesFile";
	protected final String DEVELPOMENT_SETTINGS = "developmentSettings";
	protected final String HOSTZMQSERVER = "hostzmqserver";
	protected final String HOSTZMQPUBLISHER = "hostzmqpublisher";
	protected final String STARTZMQSERVER = "startzmqserver";
	protected final String Version = "0301003";
	protected boolean autoStartZMQ = false;
	protected boolean isLinux = false;
	protected UposScale uposscale;
	protected String moduleFileName1;
	protected String moduleFileName2;
	protected String moduleFileNameZMQServer;
	protected String hostZMQServer;
	protected String hostZMQPublisher;
	protected int approvalJava;
	protected boolean autoDisable;
	protected boolean capCompareFirmwareVersion;
	protected int capPowerReporting;
	private boolean capStatusUpdate;
	private int statusNotify;
	private boolean capStatisticsReporting;
	private boolean capUpdateFirmware;
	private boolean capUpdateStatistics;
	protected String checkHealth;
	protected int dataCount;
	protected boolean dataEventEnabled;
	protected boolean deviceEnabled = false;
	protected boolean asyncMode = false;
	protected boolean freezeEvents;
	protected int powerNotify;
	protected int powerState;
	protected String physicalDeviceDescription;
	protected String physicalDeviceName;
	protected String loggingPropertiesFilePath;
	protected Vector<ScaleInternalEvent> eventsQueue;
	protected EventCallbacks eventCallbacks;
	String developmentSettings = "";
	private AsyncReaderThread asyncReader;
	private StatusUpdateDirectIOThread statusUpdater;
	private static final Logger logger = Logger.getLogger(BizerbaCSScaleService19.class.getName());;
		
	/**
	 * Constructor of the class. All attributtes are initalized to default
	 * values.
	 * 
	 */
	public BizerbaCSScaleService19() {
		
		System.setProperty("java.util.logging.SimpleFormatter.format", "[%1$tF %1$tT.%1$tL] %4$s [%2$s] %5$s%6$s%n");  
		uposscale = null;		
		eventsQueue = null;
		eventCallbacks = null;
		asyncReader = null;
		statusUpdater = null;
		autoDisable = false;
		capCompareFirmwareVersion = false;
		capPowerReporting = JposConst.JPOS_PR_NONE;
		capStatisticsReporting = false;
		capUpdateFirmware = false;
		capUpdateStatistics = false;
		capStatusUpdate = true;
		statusNotify = ScaleConst.SCAL_SN_DISABLED;
		checkHealth = "";
		dataCount = 0;
		dataEventEnabled = false;
		freezeEvents = false;
		powerNotify = JposConst.JPOS_PN_DISABLED;
		powerState = JposConst.JPOS_PS_UNKNOWN;
		physicalDeviceDescription = "";
		physicalDeviceName = "";		
		eventsQueue = new Vector<>();
		loggingPropertiesFilePath = "";
		hostZMQServer = "5558";
		hostZMQPublisher = "5559";
		moduleFileName2 = getClass().getProtectionDomain().getCodeSource().toString();
		logger.info(moduleFileName2);
		String REGEX = "file:";		
		Pattern p = Pattern.compile(REGEX);		
	    String[] items = p.split(moduleFileName2);		    
	    moduleFileName2 = items[items.length-1];	    
		int lastIndex = moduleFileName2.lastIndexOf("com_bizerba_jpos_scalecsoem.jar");
		if(lastIndex > -1)
		{
			moduleFileName2 = moduleFileName2.substring(0,lastIndex); 
			moduleFileName2 = moduleFileName2.replaceAll("%20", " ");
			moduleFileName1 = moduleFileName2;			
			moduleFileName2 += "com_bizerba_jpos_scalecsoem.jar";
			String seperator = System.getProperty("file.separator");
			logger.info("Seperator:" + seperator);
			if(seperator.charAt(0) == '/')			
			{
				logger.info("found linux system");
				isLinux = true;
				/* Linux oder unix Pfad */
				/* Pfad fuer native library unter linux */
				moduleFileName1 += "linux_x86/";
				moduleFileNameZMQServer = moduleFileName1;
				moduleFileName1 += "libjposscalejni.so";			
			}
			else
			{
				/* Pfad fuer native library unter windows */
				logger.info("Found windows system");
				if(moduleFileName1.charAt(0)=='/')
				{
					moduleFileName1 = moduleFileName1.substring(1);
					moduleFileName2 = moduleFileName2.substring(1);
				}
				moduleFileName1 += "winnt_x86\\";
				moduleFileNameZMQServer = moduleFileName1;
				moduleFileName1 += "jposscalejni.dll";
			}
		}
		else
		{						       
			//moduleFileName1 = "D:/projekte/RS/Components/SW/RIK/upos2.x/drivers/OEMScale/jposscale/lib/winnt_x86/jposscalejni.dll";
			moduleFileName1 = "";					
			moduleFileName2 = "";
			moduleFileNameZMQServer = "";
		}
		
		logger.info(moduleFileName1);
		logger.info(moduleFileName2);
		logger.info(moduleFileNameZMQServer);
	}
	
	/**
	 * This method returns the numeric code indicating device's state. The
	 * numeric codes are those defined by JavaPOS.
	 * 
	 * @see jpos.services.BaseService#getState()
	 */
	public int getState() throws JposException 
	{
		if(uposscale == null)
		{
			return JposConst.JPOS_S_CLOSED;
		}
		return uposscale.getState();
	}
	
	/**
	 * This method returns the string representing the Device Service
	 * Description
	 * 
	 * @see jpos.services.BaseService#getDeviceServiceDescription()
	 */
	public String getDeviceServiceDescription() throws JposException
	{
		checkOpened();
		return uposscale.getDeviceServiceDescription();
	}

	/**
	 * This method returns the integer value representing the version number for
	 * Device Service
	 * 
	 * @see jpos.services.BaseService#getDeviceServiceVersion()
	 */
	public int getDeviceServiceVersion() throws JposException
	{
		return 1009000;
	}

	/**
	 * Command for testing internal state of the device.
	 * 
	 * @see jpos.services.BaseService#checkHealth(int)
	 */
	public void checkHealth(int pLevel) throws JposException
	{
		checkOpened();
		uposscale.checkHealth(pLevel);
	}

	/**
	 * This method returns the string value of the CheckHealth attribute.
	 * 
	 * @see jpos.services.BaseService#getCheckHealthText()
	 */
	public String getCheckHealthText() throws JposException 
	{
		checkOpened();
		return uposscale.getHealthText();
	}

	/**
	 * This method determines whether the version of the firmware contained in
	 * the specified file is newer than, older than, or the same as the version
	 * of the firmware in the physical device. The property
	 * capCompareFirmwareVersion must be true in order to successfully use this
	 * method.
	 * 
	 * @see jpos.services.ScaleService19#compareFirmwareVersion(String, int[])
	 */
	public void compareFirmwareVersion(String arg0, int[] arg1)
			throws JposException 
	{
		if (!capCompareFirmwareVersion)
		{
			throw new JposException(JposConst.JPOS_E_ILLEGAL,
					"Compare firmware version functionality not supported");
		}
	}
	
	/**
	 * This method updates the firmware of a device with the version of the
	 * firmware contained or defined in the file specified by the parameter
	 * regardless of whether that firmware�s version is newer than, older than,
	 * or the same as the version of the firmware already in the device.
	 * 
	 * @see jpos.services.ScaleService19#updateFirmware(java.lang.String)
	 */
	public void updateFirmware(String arg0) throws JposException
	{
		if (!capUpdateFirmware)
		{
			throw new JposException(JposConst.JPOS_E_ILLEGAL,
					"Update firmware functionality not supported");
		}
	}

	/**
	 * Resets the defined resettable statistics in a device to zero. All the
	 * requested statistics must be successfully reset in order for this method
	 * to complete successfully.
	 * 
	 * Both CapStatisticsReporting and CapUpdateStatistics must be true in order
	 * to successfully use this method.
	 * 
	 * @see jpos.services.ScaleService18#resetStatistics(String)
	 */
	public void resetStatistics(String arg0) throws JposException
	{
		if (!(capStatisticsReporting && capUpdateStatistics))
		{
			throw new JposException(JposConst.JPOS_E_ILLEGAL,
					"Reset statistics functionality not supported");
		}
	}

	/**
	 * Retrieves the requested statistics from a device. The property
	 * capStatisticsReporting must be true in order to successfully use this
	 * method.
	 * 
	 * @see jpos.services.ScaleService18#retrieveStatistics(String[])
	 */
	public void retrieveStatistics(String[] arg0) throws JposException
	{
		if (!capStatisticsReporting)
		{
			throw new JposException(JposConst.JPOS_E_ILLEGAL,
					"Retrieve statistics functionality not supported");
		}
	}

	/**
	 * Updates the defined resettable statistics in a device. All the requested
	 * statistics must be successfully updated in order for this method to
	 * complete successfully.
	 * 
	 * Both CapStatisticsReporting and CapUpdateStatistics must be true in order
	 * to successfully use this method.
	 * 
	 * @see jpos.services.ScaleService18#updateStatistics(String)
	 */
	public void updateStatistics(String arg0) throws JposException
	{
		if (!(capStatisticsReporting && capUpdateStatistics))
		{
			throw new JposException(JposConst.JPOS_E_ILLEGAL,
					"Update statistics functionality not supported");
		}
	}
	
	/**
	 * This method loads the configuration information for the given logical
	 * name, then initializes the attributes of Device Service according to the
	 * physical scale characteristics.
	 * 
	 * @param logicalName -
	 *            string representing the identifier of the configuration entry
	 *            in jpos.xml file eventCallbacks - instance of EventCallbacks
	 *            class, representing the link with the Device Control calling
	 *            open method on this Device Service
	 * 
	 * @see jpos.services.BaseService#open(java.lang.String,
	 *      jpos.services.EventCallbacks)
	 */
	public void open(String logicalName, EventCallbacks pEventCallbacks)
			throws JposException 
	{
		logger.info("logical name:"+logicalName);
		

		JposEntryRegistry entryRegistry = JposServiceLoader.getManager()
				.getEntryRegistry();

		if (entryRegistry == null)
		{
			logger.info("Not found configuration registry");
			throw new JposException(JposConst.JPOS_E_NOSERVICE,
					"Configuration information not found");
		}
		
		JposEntry jposEntry = entryRegistry.getJposEntry(logicalName);
		if (jposEntry == null)
		{
			throw new JposException(JposConst.JPOS_E_NOEXIST,
					"Device configuration not found");
		}
	    
		if (jposEntry.hasPropertyWithName(LOGGING_PROPERTIES_FILE))
		{
			Object objValue = jposEntry.getPropertyValue(LOGGING_PROPERTIES_FILE);
			if (objValue != null)
			{
				loggingPropertiesFilePath = ((String) objValue).trim();
				try
				{
					// initialize logging system
					Logger.getLogger(BizerbaCSScaleService19.class.getName());
					File logConfigurationFile = new File(
							loggingPropertiesFilePath);
					if (logConfigurationFile.exists()) {
						LogManager.getLogManager().readConfiguration(
								new FileInputStream(logConfigurationFile));
					}
					else
					{
						LogManager.getLogManager().reset();
					}					
				}
				catch (IOException genEx)
				{
				}
			}
		}
		
        if (jposEntry.hasPropertyWithName(PHYSICAL_DEVICE_NAME))
        {
            Object objValue = jposEntry.getPropertyValue(PHYSICAL_DEVICE_NAME);
            if (objValue != null)
            {
            	physicalDeviceName = (String) objValue;
            } 
            else
            {
            	throw new JposException(JposConst.JPOS_E_NOEXIST, "no physical device for logical name "+logicalName+" found");
            }
        }
        else
        {
        	/* to be compatible with older XML version */
        	physicalDeviceName = logicalName;
        }

        logger.info("physical device name:"+physicalDeviceName);
        
	    if (jposEntry.hasPropertyWithName(PHYSICAL_DEVICE_DESCRIPTION))
	    {
	        Object objValue = jposEntry.getPropertyValue(PHYSICAL_DEVICE_DESCRIPTION);	
	        if (objValue != null)
	        {
	        	physicalDeviceDescription = (String) objValue;
	        } 
	        else
	        {
	        	physicalDeviceDescription = DEFAULT_PHYSICAL_DESCRIPTION;
	        }
	    }
	    
	    logger.info("physical device description:"+physicalDeviceDescription);
		
		String iniFile = "";
		if (jposEntry.hasPropertyWithName(INI_FILE_KEY))
		{
			Object objValue = jposEntry.getPropertyValue(INI_FILE_KEY);
			if (objValue != null)
			{
				iniFile = ((String) objValue).trim();
				if(iniFile.length() > 0)
				{
					logger.info("using configuration from:"+iniFile);
				}
				else
				{
					logger.info("using global configuration");
				}
			}
		}
		
		if (jposEntry.hasPropertyWithName(DEVELPOMENT_SETTINGS))
		{
			Object objValue = jposEntry.getPropertyValue(DEVELPOMENT_SETTINGS);
			if (objValue != null)
			{
				developmentSettings = ((String) objValue).trim();
				if(developmentSettings.length() > 0)
				{
					logger.info("special settings for development:"+developmentSettings);
					/* Just for debugging purpose set path to JNI  */
					moduleFileName1 = developmentSettings;
				}
				else
				{
					logger.info("using global configuration");
				}
			}
		}

		if (jposEntry.hasPropertyWithName(HOSTZMQSERVER))
		{
			String zmqHost = "";
			Object objValue = jposEntry.getPropertyValue(HOSTZMQSERVER);
			if (objValue != null)
			{
				zmqHost = ((String) objValue).trim();
				if(zmqHost.length() > 0)
				{
					logger.info("use host port from xml file");					
					hostZMQServer = zmqHost;
				}
			}
		}
		
		if (jposEntry.hasPropertyWithName(HOSTZMQPUBLISHER))
		{
			String zmqHost = "";
			Object objValue = jposEntry.getPropertyValue(HOSTZMQPUBLISHER);
			if (objValue != null)
			{
				zmqHost = ((String) objValue).trim();
				if(zmqHost.length() > 0)
				{
					logger.info("use publisher port from xml file");					
					hostZMQPublisher = zmqHost;
				}
			}
		}
		
		if (jposEntry.hasPropertyWithName(STARTZMQSERVER))
		{
			Object objValue = jposEntry.getPropertyValue(STARTZMQSERVER);
			if (objValue != null)
			{
				String value = ((String) objValue).trim();
				if(value.length() > 0)
				{
					if(value.matches("true"))
					{
						autoStartZMQ = true;
					}
					else
					{
						autoStartZMQ = false;
					}										
				}
				else
				{
					autoStartZMQ = false;
				}				
			}
			else
			{
				autoStartZMQ = false;
			}
		}
		
		if(moduleFileName1.length() == 0)
		{
			throw new JposException(JposConst.JPOS_E_NOSERVICE,
			"No valid modules found");
		}
		if(moduleFileName1.endsWith(".exe") || moduleFileName1.endsWith(".x"))
		{
			/* Server fuer ZMQ und Protokollbuffer starten (gemischte Anbindung 32 und 64 bit moeglich) */
			logger.info("Host port to ZMQ Server:"+ hostZMQServer);
			logger.info("Publisher port to ZMQ Server:"+ hostZMQPublisher);			
			if(autoStartZMQ == true)
			{
				logger.info("start zmq server with open");
			}
			else
			{
				logger.info("zmq server not start with open");
			}	
			uposscale = UposScale.getInstance(moduleFileName1,moduleFileName2, approvalJava, iniFile, this, true, autoStartZMQ, hostZMQServer, hostZMQPublisher,Version);
		}
		else
		{
			/* Anbindung im gleichen Prozessraum (nur bei 32 bit moeglich) via dll,so */
			uposscale = UposScale.getInstance(moduleFileName1,moduleFileName2, approvalJava, iniFile, this, false, false,"", "",Version);
		}
		try
		{
			uposscale.open(physicalDeviceName);
		}
		catch (JposException e)
		{
			uposscale.releaseInstance();
			throw e;
		}
		autoDisable = false;
		checkHealth = "";
		capPowerReporting = JposConst.JPOS_PR_NONE;
		eventsQueue.clear();
		dataCount = 0;
		dataEventEnabled = false;
		freezeEvents = false;
		powerNotify = JposConst.JPOS_PN_DISABLED;
		powerState = JposConst.JPOS_PS_UNKNOWN;		
		eventCallbacks = pEventCallbacks;
	}
	
	/**
	 * This method returns the string value representing description for the
	 * physical scale device
	 * 
	 * @see jpos.services.BaseService#getPhysicalDeviceDescription()
	 */
	public String getPhysicalDeviceDescription() throws JposException
	{
		checkOpened();
		return physicalDeviceDescription;
	}

	/**
	 * This method returns the string value representing the name of physical
	 * device
	 * 
	 * @see jpos.services.BaseService#getPhysicalDeviceName()
	 */
	public String getPhysicalDeviceName() throws JposException
	{
		checkOpened();
		return physicalDeviceName;
	}
	
	/**
	 * This method returns the value of the CapPowerReporting attribute.
	 * 
	 * @see jpos.services.ScaleService13#getCapPowerReporting()
	 */
	public int getCapPowerReporting() throws JposException
	{
		checkOpened();
		return capPowerReporting;
	}
	
	/**
	 * This method reports the value of CapCompareFirmwareVersion attribute.
	 * 
	 * @return true, if the device supports comparing the version of the
	 *         firmware in the physical device against that of a firmware file
	 *         via {@link #compareFirmwareVersion(String, int[])}.
	 * 
	 * @see jpos.services.ScaleService19#getCapCompareFirmwareVersion()
	 */
	public boolean getCapCompareFirmwareVersion() throws JposException
	{
		checkOpened();
		return true;		
	}

	/**
	 * This method reports the value of CapStatusUpdate attribute.
	 * 
	 * @return true, then the scale is capable of providing scale weight status
	 *         with #{@link jpos.events.StatusUpdateEvent}s. This property is
	 *         initialized by the open method. If true when the device is
	 *         enabled, an immediate #{@link jpos.events.StatusUpdateEvent}
	 *         will be generated to tell the application the current state of
	 *         the scale.
	 * 
	 * @see jpos.services.ScaleService19#getCapStatusUpdate()
	 */
	public boolean getCapStatusUpdate() throws JposException
	{
		checkOpened();
		return capStatusUpdate;		
	}

	/**
	 * This method reports the value of CapUpdateFirmware attribute.
	 * 
	 * @return true, if the device�s firmware can be updated via the
	 *         {@link #updateFirmware(String)} method.
	 * 
	 * @see jpos.services.ScaleService19#getCapUpdateFirmware()
	 */
	public boolean getCapUpdateFirmware() throws JposException
	{
		checkOpened();
		return capUpdateFirmware;
	}
	
	/**
	 * This method reports the value of CapStatisticsReporting attribute.
	 * 
	 * @return true, if the device accumulates and can provide various
	 *         statistics regarding usage, otherwise no usage statistics are
	 *         accumulated.
	 * 
	 * @see jpos.services.ScaleService18#getCapStatisticsReporting()
	 */
	public boolean getCapStatisticsReporting() throws JposException
	{
		checkOpened();		
		return capStatisticsReporting;
	}

	/**
	 * This method reports the value of CapUpdateStatistics attribute.
	 * 
	 * @return true, if the device statistics, or some of the statistics, can be
	 *         reset to zero using the {@link #resetStatistics(String)} method,
	 *         or updated using the {@link #updateStatistics(String)} method.
	 * 
	 * @see jpos.services.ScaleService18#getCapUpdateStatistics()
	 */
	public boolean getCapUpdateStatistics() throws JposException
	{
		checkOpened();		
		return capUpdateStatistics;
	}
	
	/**
	 * This method reports the value of CapDisplay attribute.
	 * 
	 * @return boolean value indicating if the scale the capability to display
	 *         description of the item being weighted.
	 * 
	 * @see jpos.services.ScaleService12#getCapDisplay()
	 */
	public boolean getCapDisplay() throws JposException
	{
		checkOpened();		
		return uposscale.capDisplay();
	}

	/**
	 * This method returns the value of the CapDisplayText attribute.
	 * 
	 * @see jpos.services.ScaleService13#getCapDisplayText()
	 */
	public boolean getCapDisplayText() throws JposException
	{
		checkOpened();		
		return uposscale.capDisplayText();		
	}

	/**
	 * This method returns the value of the CapPriceCalculating attribute.
	 * 
	 * @see jpos.services.ScaleService13#getCapPriceCalculating()
	 */
	public boolean getCapPriceCalculating() throws JposException
	{
		checkOpened();		
		return uposscale.capPriceCalculating();		
	}

	/**
	 * This method returns the value of the CapTareWeight attribute.
	 * 
	 * @see jpos.services.ScaleService13#getCapTareWeight()
	 */
	public boolean getCapTareWeight() throws JposException
	{
		checkOpened();		
		return uposscale.capTareWeight();	
	}

	/**
	 * This method returns the value of the CapZeroScale attribute.
	 * 
	 * @see jpos.services.ScaleService13#getCapZeroScale()
	 */
	public boolean getCapZeroScale() throws JposException
	{
		checkOpened();		
		return uposscale.capZeroScale();	
	}
	
	/**
	 * Scale weight state notification can only be set by the application if the
	 * capability CapStatusUpdate is true. StatusNotify may only be set while
	 * the device is disabled, that is, while DeviceEnabled is false.
	 * 
	 * @see jpos.services.ScaleService19#setStatusNotify(int)
	 */
	public void setStatusNotify(int arg0) throws JposException 
	{
		checkOpened();
		if (!capStatusUpdate) {
			throw new JposException(JposConst.JPOS_E_ILLEGAL,
					"Unable to set StatusNotify while CapStatusUpdate is false");
		}
		if (uposscale.getDeviceEnabled())
		{
			throw new JposException(JposConst.JPOS_E_ILLEGAL,
					"Unable to set StatusNotify while DeviceEnabled is true");
		}
		statusNotify = arg0;
	}
	
	/**
	 * This method reports the value of StatusNotify attribute.
	 * 
	 * @return #{@link jpos.ScaleConst} SCAL_SN_DISABLED or SCAL_SN_ENABLED
	 * 
	 * @see jpos.services.ScaleService19#getStatusNotify()
	 */
	public int getStatusNotify() throws JposException
	{
		checkOpened();
		return statusNotify;
	}

	/**
	 * 
	 * @see jpos.services.ScaleService13#setPowerNotify(int)
	 */
	public void setPowerNotify(int param) throws JposException
	{
		checkOpened();
		if (capPowerReporting == JposConst.JPOS_PR_NONE) {
			throw new JposException(JposConst.JPOS_E_ILLEGAL,
					"Power Notification not supported");
		}
		if (uposscale.getDeviceEnabled())
		{
			throw new JposException(JposConst.JPOS_E_ILLEGAL,
			"Device is already enabled");
		}
		// set implicit value
		powerNotify = param;
	}
	
	/**
	 * This method returns the value of the PowerNotify attribute.
	 * 
	 * @return integer value according to JavaPOS standard
	 * @see jpos.services.ScaleService13#getPowerNotify()
	 */
	public int getPowerNotify() throws JposException
	{
		checkOpened();
		return powerNotify;
	}
	
	/**
	 * This method returns the value of the PowerState attribute.
	 * 
	 * @return integer value according to JavaPOS standard
	 * @see jpos.services.ScaleService13#getPowerState()
	 */
	public int getPowerState() throws JposException
	{
		checkOpened();		
		return powerState;
	}
	
	/**
	 * This method returns the maximum number of characters allowed in the text
	 * to be displayed on scale's display, if the scale supports such
	 * functionality.
	 * 
	 * @see jpos.services.ScaleService13#getMaxDisplayTextChars()
	 */
	public int getMaxDisplayTextChars() throws JposException
	{
		checkOpened();
		return uposscale.getMaxDisplayTextChars();
	}

	/**
	 * This method returns the integer value indicating the maximum weight
	 * allowed for this scale device
	 * 
	 * @see jpos.services.ScaleService12#getMaximumWeight()
	 */
	public int getMaximumWeight() throws JposException
	{
		checkOpened();
		return uposscale.getMaximumWeight();		
	}
	
	/**
	 * This method requests the exclusive access to the scale device. Due to the
	 * fact that the Scale is connected on a serial port and when an application
	 * opens a communicaton connection to a serial port in fact that application
	 * becomes the owner of the port, then inside this methods is effectivelly
	 * opened the communication connection to the port where the scale is
	 * physically connected on. Moreover, implementation of this methods
	 * complies to JavaPOS specifications.
	 * 
	 * @see jpos.services.BaseService#claim(int)
	 */
	public void claim(int pTimeout) throws JposException 
	{
		checkOpened();
		uposscale.claim(pTimeout);		
		asyncReader = new AsyncReaderThread();
		statusUpdater = new StatusUpdateDirectIOThread();
		logger.info("Created and Started thread for asynchronous reading from scale device.");
	}

	/**
	 * This method returns the boolean value indicating if the device is claimed
	 * or not.
	 * 
	 * @return boolean value
	 * 
	 * @see jpos.services.BaseService#getClaimed()
	 */
	public boolean getClaimed() throws JposException
	{
		checkOpened();
		return uposscale.getClaimed();		
	}
	
	/**
	 * 
	 * @see jpos.services.BaseService#setDeviceEnabled(boolean)
	 */
	public void setDeviceEnabled(boolean flag) throws JposException
	{
		checkOpened();
		uposscale.setDeviceEnabled(flag);
		deviceEnabled = flag;
		logger.info("Forwarded the command to setDeviceEnable to: " + flag);
	}

	/**
	 * This method returns the value of the DeviceEnabled attribute.
	 * 
	 * @return boolean value
	 * 
	 * @see jpos.services.BaseService#getDeviceEnabled()
	 */
	public boolean getDeviceEnabled() throws JposException
	{
		checkOpened();
		return deviceEnabled;
	}
	
	/**
	 * This method implements the release operation for this Device Service.
	 * 
	 * @see jpos.services.BaseService#release()
	 */
	public void release() throws JposException 
	{
		checkOpened();
		if (asyncReader != null) {
			logger.info("Stop eventual asynchronous reading and stop the thread-reader too");
			asyncReader.stopRead();
			try {
				asyncReader.interrupt();
				asyncReader.kill();
				asyncReader = null;
				logger.info("Asynchronous reader set to NULL");
			} catch (Exception ex) {
			}
		}
		if(statusUpdater != null)
		{
			logger.info("Stop statusUpdater thread");
			try {
				statusUpdater.interrupt();
				statusUpdater.kill();
				statusUpdater = null;
				logger.info("statusUpdater set to NULL");
			} catch (Exception ex) {
			}	
		}
		logger.info("Command for releasing Device");
		uposscale.releaseDevice();
		logger.info("Clear events queue");
		clearInput();
	}
	
	/**
	 * Through this method the device is first disabled, then it is released and
	 * in this mode is closed also the communication connection with the serial
	 * port which on the scale device is conneted.
	 * 
	 * @see jpos.services.BaseService#close()
	 */
	public void close() throws JposException
	{
		checkOpened();
		if (getClaimed())
		{
			if (getDeviceEnabled())
			{
				logger.info("Disable device");
				setDeviceEnabled(false);
			}
			logger.info("Release device and the connection to the scale");
			release();
		}
    	uposscale.close();
    	uposscale.releaseInstance();
    	uposscale = null;
	}
	
	/**
	 * This method reports the value of AsyncMode attribute.
	 * 
	 * @return boolean value indicating if the AyncMode is On or Off.
	 * 
	 * @see jpos.services.ScaleService13#getAsyncMode()
	 */
	public boolean getAsyncMode() throws JposException 
	{
		checkOpened();
		return asyncMode;
	}

	/**
	 * This method allows to set AsyncMode attribute to TRUE/FALSE, in order to
	 * indicate to indicate to the driver to respond synchronous / asynchronous
	 * 
	 * @see jpos.services.ScaleService13#setAsyncMode(boolean)
	 */
	public void setAsyncMode(boolean pMode) throws JposException
	{
		checkOpened();
		uposscale.setAsyncMode(pMode);
		asyncMode = pMode;
	}

	/**
	 * This method allows to set on TRUE/FALSE the attribute that indicates to
	 * Device Service whether to disable or not the device after a DataEvent is
	 * received and put into the queue.
	 * 
	 * @see jpos.services.ScaleService13#setAutoDisable(boolean)
	 */
	public void setAutoDisable(boolean flag) throws JposException
	{
		checkOpened();
		autoDisable = flag;
		logger.info("Set AutoDisable to: " + flag);
	}
	
	/**
	 * This method reports the value of AutoDisable attribute.
	 * 
	 * @return boolean value indicating if the AutoDisable is On or Off.
	 * 
	 * @see jpos.services.ScaleService13#getAutoDisable()
	 */
	public boolean getAutoDisable() throws JposException
	{
		checkOpened();
		return autoDisable;
	}

	/**
	 * 
	 * @see jpos.services.ScaleService13#setDataEventEnabled(boolean)
	 */
	public void setDataEventEnabled(boolean flag) throws JposException
	{
		checkOpened();
		dataEventEnabled = flag;
		logger.info("Set DataEventEnabled to: " + flag);
		if (dataEventEnabled && !getFreezeEvents() && eventsQueue.size() > 0)
		{
			logger.info("Try to send events");
			sendEvent();
		}
	}

	/**
	 * This method returns the value of the DataEventEnabled attribute.
	 * 
	 * @return boolean value
	 * 
	 * @see jpos.services.ScaleService13#getDataEventEnabled()
	 */
	public boolean getDataEventEnabled() throws JposException
	{
		checkOpened();
		return dataEventEnabled;	
	}
	
	/**
	 * 
	 * @see jpos.services.BaseService#setFreezeEvents(boolean)
	 */
	public void setFreezeEvents(boolean flag) throws JposException
	{
		checkOpened();
		freezeEvents = flag;
		if (!freezeEvents && getDataEventEnabled() && eventsQueue.size() > 0) {
			logger.info("Try to send events");
			sendEvent();
		}
	}
	
	/**
	 * This method returns the value of the FreezeEvents attribute.
	 * 
	 * @return boolean value
	 * @see jpos.services.BaseService#getFreezeEvents()
	 */
	public boolean getFreezeEvents() throws JposException
	{
		checkOpened();
		return freezeEvents;
	}
	
	/**
	 * This method returns the value of DataCount attribute.
	 * 
	 * @return integer value
	 * 
	 * @see jpos.services.ScaleService13#getDataCount()
	 */
	public int getDataCount() throws JposException
	{
		checkOpened();
		return dataCount;		
	}

	/**
	 * This method stops the still executing asynchronous reading and then clear
	 * the events queue, setting the DataCount attribut to 0.
	 * 
	 * @see jpos.services.ScaleService13#clearInput()
	 */
	public void clearInput() throws JposException
	{
		if (asyncReader != null && asyncReader.isRunning())
		{
			asyncReader.stopRead();
			logger.info("Stop asynchrounous reading");
		}

		synchronized (eventsQueue)
		{
			eventsQueue.clear();
			dataCount = eventsQueue.size();
		}
		logger.info("Cleared events queue");
	}

	/**
	 * This method allows to post an internal event corresponding to a DataEvent
	 * into the events queue and if the context is appropriate sends the first
	 * event from the queue.
	 * 
	 * @param pData -
	 *            instance containing the information returned by the scale
	 */
	public void postDataEvent(ScaleDataResponse pData)
	{
		if (eventsQueue == null || pData == null)
		{
			return;
		}
		try
		{
			if (getDeviceEnabled())
			{
				ScaleInternalEvent internalEvent = new ScaleInternalEvent(
						ScaleInternalEvent.DATA_TYPE, pData);
				logger.info("Prepared internal event using information from ScaleDataResponse");
				synchronized (eventsQueue) {
					eventsQueue.add(internalEvent);
					dataCount++;
				}
				logger.info("Added internal event to the queue");
			}
			else
			{
				logger.info("Device is disabled, so discard event");
				return;
			}

			if (!getFreezeEvents() && getDataEventEnabled())
			{
				logger.info("Try to send event");
				sendEvent();
				if (getAutoDisable()) {
					logger.info("Disable device");
					setDeviceEnabled(false);
				}
			}
		} 
		catch (JposException ex)
		{
			logger.log(Level.SEVERE, "Exception occured", ex);
		}
	}

	/**
	 * This method allows to post an internal event corresponding to an
	 * ErrorEvnt into the events queue and if the context is appropriate sends
	 * the first event from the queue.
	 * 
	 * @param -
	 *            error eventalready prepared
	 */
	public void postErrorEvent(ErrorEvent pEvent)
	{
		if (eventsQueue == null || pEvent == null)
		{
			return;
		}
		try
		{
			if (!getDeviceEnabled()) {
				logger.info("Device disabled, event discarded");
				return;
			}
		}
		catch (JposException ex)
		{
			logger.log(Level.SEVERE, "Device closed or not claimed, event discarded", ex);
			return;
		}

		ScaleInternalEvent internalEvent = new ScaleInternalEvent(
				ScaleInternalEvent.ERROR_TYPE, pEvent);
		logger.info("Prepared internal event corresponding to the error event");

		synchronized (eventsQueue)
		{
			if (pEvent.getErrorLocus() == JposConst.JPOS_EL_INPUT_DATA)
			{
				logger.info("Error Event with locus INPUT_DATA added first in queue");
				eventsQueue.add(0, internalEvent);

				ErrorEvent auxErrEvent = new ErrorEvent(pEvent.getSource(),
						pEvent.getErrorCode(), pEvent.getErrorCodeExtended(),
						JposConst.JPOS_EL_INPUT, JposConst.JPOS_ER_CLEAR);
				ScaleInternalEvent auxInternEvent = new ScaleInternalEvent(
						ScaleInternalEvent.ERROR_TYPE, auxErrEvent);
				// eventsQueue.add(internalEvent);
				eventsQueue.add(auxInternEvent);
				logger.info("Aux Error Event with locus INPUT_DATA added");
			}
			else
			{
				eventsQueue.add(internalEvent);
			}
		}
		logger.info("[" + Thread.currentThread().getName() + ": "
				+ Thread.currentThread().hashCode()
				+ "] - added event to the queue");

		try
		{
			if (!getFreezeEvents() && getDataEventEnabled())
			{
				logger.info("Try to send event");
				sendEvent();
				if (getAutoDisable()) {
					logger.info("Auto Disable device");
					setDeviceEnabled(false);
				}
			}
		}
		catch (JposException ex)
		{
			logger.log(Level.SEVERE, "JposException", ex);
		}
	}
	
	/**
	 * This method allows to post an internal event corresponding to a StatusUpdateEvent
	 * into the events queue and if the context is appropriate sends the first
	 * event from the queue.
	 * 
	 * @param pData -
	 *            instance containing the information returned by the scale
	 */
	public void postStatusUpdateEvent(ScaleUpdateResponse pData)
	{
		if (eventsQueue == null || pData == null)
		{
			return;
		}
		try
		{
			if (getDeviceEnabled())
			{
				ScaleInternalEvent internalEvent = new ScaleInternalEvent(
						ScaleInternalEvent.STATUS_TYPE, pData);
				logger.info("Prepared internal event using information from ScaleUpdateResponse");
				synchronized (eventsQueue) {
					eventsQueue.add(internalEvent);					
				}
				logger.info("Added internal event to the queue");
			}
			else
			{
				logger.info("Device is disabled, so discard event");
				return;
			}

			if (!getFreezeEvents() && (getStatusNotify() == ScaleConst.SCAL_SN_ENABLED) )
			{
				statusUpdater.fireQueue();
				if (getAutoDisable()) {
					logger.info("Disable device");
					setDeviceEnabled(false);
				}
			}
		} 
		catch (JposException ex)
		{
			logger.log(Level.SEVERE, "Exception occured", ex);
		}
	}
	
	/**
	 * This method allows to post an internal event corresponding to a DirectIOEvent
	 * into the events queue and if the context is appropriate sends the first
	 * event from the queue.
	 * 
	 * @param pData -
	 *            instance containing the information returned by the scale
	 */
	public void postDirectIOEvent(ScaleDirectIOResponse pData)
	{
		if (eventsQueue == null || pData == null)
		{
			return;
		}
		try
		{
			if (getDeviceEnabled())
			{
				ScaleInternalEvent internalEvent = new ScaleInternalEvent(
						ScaleInternalEvent.DIRECTIO_TYPE, pData);
				logger.info("Prepared internal event using information from ScaleDirectIOResponse");
				synchronized (eventsQueue) {
					eventsQueue.add(internalEvent);					
				}
				logger.info("Added internal event to the queue");
			}
			else
			{
				logger.info("Device is disabled, so discard event");
				return;
			}

			if (!getFreezeEvents())
			{
				statusUpdater.fireQueue();
				if (getAutoDisable()) {
					logger.info("Disable device");
					setDeviceEnabled(false);
				}
			}
		} 
		catch (JposException ex)
		{
			logger.log(Level.SEVERE, "Exception occured", ex);
		}
	}

	

	/**
	 * This method performs actions for sending the first event from the queue
	 * to the Device Control through the EventCallback instance
	 */
	private void sendEvent()
	{
		ScaleInternalEvent internalEvent = null;

		if (eventsQueue.size() > 0)
		{
			synchronized (eventsQueue)
			{
				logger.info("There are (" + eventsQueue.size()
						+ ") events in the queue");
				internalEvent = (ScaleInternalEvent) eventsQueue.remove(0);
			}

			Object temp = internalEvent.getInfo();

			switch (internalEvent.getType())
			{
				case ScaleInternalEvent.DATA_TYPE:
				{
					if (temp != null)
					{
						logger.info("Prepare DataEvent to send");
						ScaleDataResponse data = (ScaleDataResponse) temp;
	
						int weight = data.weightValue;
						DataEvent dataEvent = new DataEvent(eventCallbacks
								.getEventSource(), weight);
	
						eventCallbacks.fireDataEvent(dataEvent);
						logger.info("DataEvent sent");
						try {
							logger.info("Disable Data Event");
							setDataEventEnabled(false);
						} catch (Exception ex) {
							logger.log(Level.SEVERE, "Exception occured", ex);
						}
	
						dataCount--;
					}
					break;
				}
				case ScaleInternalEvent.ERROR_TYPE:
				{
					if (temp != null) {
						ErrorEvent event = (ErrorEvent) temp;
						eventCallbacks.fireErrorEvent(event);
						logger.info("ErrorEvent sent");
						if (event.getErrorResponse() == JposConst.JPOS_ER_CLEAR) {
							try 
							{
								clearInput();
								setDataEventEnabled(false);
							}
							catch (Exception ex1)
							{
							}
						} 
						else if (event.getErrorResponse() == JposConst.JPOS_ER_CONTINUEINPUT
								&& event.getErrorLocus() == JposConst.JPOS_EL_INPUT_DATA)
						{
	
						}
						else if (event.getErrorResponse() == JposConst.JPOS_ER_RETRY
								&& event.getErrorLocus() == JposConst.JPOS_EL_INPUT)
						{
							try
							{
								if (!getDeviceEnabled())
								{
									logger.info("Found device disabled");
									break;
								}
							} catch (Exception ex)
							{
								logger.log(Level.SEVERE, "Found device disabled", ex);
								break;
							}
						}
					}
					break;
				}
				case ScaleInternalEvent.STATUS_TYPE:
				{
					ScaleUpdateResponse updateResponse = (ScaleUpdateResponse)temp;					
					StatusUpdateEvent firedEvent = new StatusUpdateEvent(eventCallbacks.getEventSource(), updateResponse.statusUpdate);
					eventCallbacks.fireStatusUpdateEvent(firedEvent);
					break;
				}
				case ScaleInternalEvent.DIRECTIO_TYPE:
				{
					ScaleDirectIOResponse directIOResponse = (ScaleDirectIOResponse)temp;					
					DirectIOEvent directIOEvent = new DirectIOEvent(eventCallbacks.getEventSource(),directIOResponse.eventNumber,directIOResponse.data, directIOResponse.obj);
					eventCallbacks.fireDirectIOEvent(directIOEvent);					
					break;
				}
				
			}
		}
	}
	
	/**
	 * inner class implementing the thread for asynchronous read from scale
	 */	
	protected class AsyncReaderThread extends Thread
	{
		int weight[];
		long timeout;
		boolean blnRunning;
		Semaphore objLocalSemaphore;		
		boolean blnQueue;
		boolean blnMustRead;
		boolean blnMustKillIt;

		AsyncReaderThread()
		{
			weight = new int[1];
			blnRunning = false;

			objLocalSemaphore = new Semaphore(0);			
			blnMustRead = false;
			blnMustKillIt = false;
			start();
		}

		void startRead(long pTimeout) {
			timeout = pTimeout;
			blnMustRead = true;

			objLocalSemaphore.release();

		}

		void stopRead() 
		{
			blnMustRead = false;
			if(blnRunning && uposscale != null)
			{
				try
				{
					uposscale.killActualCommand();
				}
				catch(Exception genEx)
				{
				}				
			}
			objLocalSemaphore.release();
		}	

		public boolean isRunning() {
			return blnRunning;
		}

		public void kill() {
			blnMustKillIt = true;
			objLocalSemaphore.release();
		}

		/**
		 * Run asynchron reading of the weight
		 */
		public void run() 
		{
			while (!blnMustKillIt)
			{
				try 
				{
					/* The semaphore allows only one time to run through the loop 
					 with the next startRead() the semaphore is released for
					 the next loop */					
					objLocalSemaphore.acquire();

					if (blnMustRead ) 
					{
						blnRunning = true;
						// read
						ScaleDataResponse dataAnswer = new ScaleDataResponse();
						try
						{
							if(uposscale != null)
							{
								dataAnswer.weightValue = uposscale.readWeight(timeout);
								postDataEvent(dataAnswer);
							}
						}
						catch (JposException ex)
						{
							 ErrorEvent errorEvent;
							//setState(JposConst.JPOS_S_ERROR);
							 if (getDataCount() > 0) 
							 {
								errorEvent = new ErrorEvent(eventCallbacks
										.getEventSource(),
										ex.getErrorCode(),
										ex.getErrorCodeExtended(),							          
										JposConst.JPOS_EL_INPUT_DATA,
										JposConst.JPOS_ER_CONTINUEINPUT);
							 }
							 else
							 {
								errorEvent = new ErrorEvent(eventCallbacks
										.getEventSource(),
										ex.getErrorCode(),
										ex.getErrorCodeExtended(),							          
										JposConst.JPOS_EL_INPUT,
										JposConst.JPOS_ER_CLEAR);
							  }
							  postErrorEvent(errorEvent);
						}
					}
				} 
				catch (Exception genEx)
				{
					// ignore exception
					logger.log(Level.SEVERE, "Exception occured", genEx);
				}
				blnMustRead = false;
				blnRunning = false;
			}
		}
	}
	
	/**
	 * inner class implementing the thread for asynchronous read from scale
	 */	
	protected class StatusUpdateDirectIOThread extends Thread
	{
		Semaphore objLocalSemaphore;		
		boolean blnQueue;
		boolean blnMustKillIt;

		StatusUpdateDirectIOThread()
		{
			objLocalSemaphore = new Semaphore(0);			
			blnMustKillIt = false;
			start();
		}
		
		void fireQueue()
		{
			objLocalSemaphore.release();
		}

		public void kill() {
			blnMustKillIt = true;
			objLocalSemaphore.release();
		}

		/**
		 * Run asynchron reading of the weight
		 */
		public void run() 
		{
			while (!blnMustKillIt)
			{
				try 
				{
					/* The semaphore allows only one time to run through the loop 
					 with the next startRead() the semaphore is released for
					 the next loop */					
					objLocalSemaphore.acquire();
					sendEvent();
				} 
				catch (Exception genEx)
				{
					// ignore exception
					logger.log(Level.SEVERE, "Exception occured", genEx);
				}
			}
		}
	}

	
	/**
	 * This method allows to send to the scale commands that are not included by
	 * JavaPOS standard.
	 * 
	 * @see jpos.services.BaseService#directIO(int, int[], java.lang.Object)
	 */
	public void directIO(int pCommand, int[] pData, Object pAuxData)
			throws JposException
	{
		if (asyncReader.isRunning() && pCommand != BizerbaScaleConstants.BUSCI_DIO_REQUEST_GET_PREC_LAST_REG )
		{
			throw new JposException(JposConst.JPOS_E_ILLEGAL,
					"Asynchronous read in progress");
		}
		checkOpened();
		long[] data = new long[1];
    	data[0] = pData[0];
    	String[] auxData;
    	if (null==pAuxData)
    	{
    		auxData=new String[1];
    		auxData[0]=new String("");
    	}
    	else
    	{
    		if (pAuxData.getClass().isArray())
    		{
    			auxData = (String[])pAuxData;
    			if (auxData[0]==null)
    				auxData[0]=new String("");    			
    		}
    		else
    		{
    			auxData= new String[1];
    			auxData[0] = (String)pAuxData;
    		}
    	}
    	uposscale.directIO(pCommand, data, auxData);
    	pData[0] = (int) data[0];
	}

	/**
	 * This method returns the value of weight unit attribute.
	 * 
	 * @see jpos.services.ScaleService12#getWeightUnit()
	 */
	public int getWeightUnit() throws JposException
	{
		checkOpened();
		return uposscale.getWeightUnit();
	}
	
	/**
	 * 
	 * @see jpos.services.ScaleService13#zeroScale()
	 */
	public void zeroScale() throws JposException
	{
		checkOpened();
		uposscale.zeroScale();
	}
	
	/**
	 * 
	 * @see jpos.services.ScaleService13#displayText(java.lang.String)
	 */
	public void displayText(String text) throws JposException
	{
		checkOpened();
		uposscale.setDisplayText(text);
	}

	/**
	 * 
	 * @see jpos.services.ScaleService13#setUnitPrice(long)
	 */
	public void setUnitPrice(long pUnitPrice) throws JposException
	{
		checkOpened();
		uposscale.setUnitPrice(pUnitPrice);
	}

	/**
	 * This method returns the value set for Unit Price attribute
	 * 
	 * @see jpos.services.ScaleService13#getUnitPrice()
	 */
	public long getUnitPrice() throws JposException
	{
		checkOpened();
		return uposscale.getUnitPrice();
	}

	/**
	 * 
	 * @see jpos.services.ScaleService13#setTareWeight(int)
	 */
	public void setTareWeight(int pTareWeight) throws JposException
	{		
		checkOpened();
		uposscale.setTareWeight(pTareWeight);		
	}
	
	/**
	 * This method returns the value set for the tare attribute.
	 * 
	 * @see jpos.services.ScaleService13#getTareWeight()
	 */
	public int getTareWeight() throws JposException
	{
		checkOpened();
		return uposscale.getTareWeight(); 
	}

	/**
	 * This method starts the weighing operation, sending to the scale device
	 * the unit price and tare weight, and reads the weighing result, if it is a
	 * synchronous reading.
	 * 
	 * @param pWeight -
	 *            array of integers, representing an output parameter where the
	 *            weighing result is placed pTimeout - integer value
	 *            representing the number of milliseconds to wait for weighing
	 *            operation's completion
	 * 
	 * @see jpos.services.ScaleService12#readWeight(int[], int)
	 */
	public void readWeight(int[] pWeight, int pTimeout) throws JposException
	{
		checkOpened();
		if (pWeight == null)
		{
			throw new JposException(JposConst.JPOS_E_ILLEGAL,
					"Array parameter for weight value is null");
		}

		if (asyncReader.isRunning())
		{
			throw new JposException(JposConst.JPOS_E_ILLEGAL,
					"Asynchronous read in progress");
		}

		if(uposscale.getAsyncMode())
		{
			if (asyncReader != null)
			{
				logger.info("Start asynchronous reading");
				asyncReader.startRead(pTimeout);
			} 
			else 
			{
				throw new JposException(JposConst.JPOS_E_FAILURE,
						"Unable to start asynchronous read");
			}
			pWeight[0] = 0;			
		}
		else
		{
			pWeight[0] = uposscale.readWeight(pTimeout);			
		}		
	}
	
	/**
	 * This method returns the long value representing the calculated price for
	 * the weighted item, based on the provided unit price.
	 * 
	 * @see jpos.services.ScaleService13#getSalesPrice()
	 */
	public long getSalesPrice() throws JposException
	{
		checkOpened();
		return uposscale.getSalesPrice();
	}

	/**
	 * Contains the returned value for the weight measured by the scale if the #{@link jpos.events.StatusUpdateEvent}
	 * status is set to #{@link jpos.ScaleConst} SCL_SUE_STABLE_WEIGHT, else
	 * zero. The weight has an assumed decimal place located after the
	 * �thousands� digit position. For example, an actual value of 12345
	 * represents 12.345, and an actual value of 5 represents 0.005. It is
	 * suggested that an application use the weight in this property only for
	 * display purposes. For a weight to use for sale purposes, it is suggested
	 * that the application call {@link #readWeight(int[], int)}.
	 * 
	 * @see jpos.services.ScaleService19#getScaleLiveWeight()
	 */
	public int getScaleLiveWeight() throws JposException
	{
		checkOpened();
		return uposscale.scaleLiveWeight();
	}

	/**
	 * @see jpos.loader.JposServiceInstance#deleteInstance()
	 */
	public void deleteInstance() throws JposException
	{
	}

	/**
	 * @see java.lang.Object#finalize()
	 */
	protected void finalize() throws Throwable {
		try 
		{
			if (asyncReader != null)
			{
					if (asyncReader.isAlive())
					{
						asyncReader.interrupt();
					}
					asyncReader.kill();
					asyncReader = null;
			}
			eventsQueue.clear();
			if(statusUpdater != null)
			{
				if (statusUpdater.isAlive())
				{
					statusUpdater.interrupt();
				}
				statusUpdater.kill();
				statusUpdater = null;				
			}
		}		
		finally
		{
			super.finalize();
		}
	}
	
	/**
	 * @throws JposException in case scale is closed
	 */
	protected void checkOpened() throws JposException
	{
		if(uposscale == null)
		{
			throw new JposException(JposConst.JPOS_E_CLOSED, "Device is closed");
		}
	}
	
	public void updateDTCCS(String dtccs, int uposStatus)
	{		
		if(statusNotify == ScaleConst.SCAL_SN_ENABLED)
		{
			/* Use the UPOS 1.14 conform status update */
			ScaleUpdateResponse statusUpdateRespons = new ScaleUpdateResponse();
			statusUpdateRespons.statusUpdate = uposStatus;
			postStatusUpdateEvent(statusUpdateRespons);			
		}
		else
		{
			/* Use the directIO event to transfer all information for the Bizerba specific DTCCS weighing information */
			ScaleDirectIOResponse pDTCCS = new ScaleDirectIOResponse();
			pDTCCS.eventNumber = BizerbaScaleConstants.BUSCI_DIO_EVENT_UPDATE_DTCCS;
			pDTCCS.data = uposStatus;
			pDTCCS.obj = dtccs;		
			postDirectIOEvent(pDTCCS);
		}
	}
	
}
